# Integração com Stripe - Sistema de Bolão

## Configuração Necessária

### 1. Instalar a biblioteca do Stripe

```bash
composer require stripe/stripe-php
```

### 2. Configurar as chaves do Stripe

No arquivo `includes/config.php`, substitua as chaves de teste pelas suas chaves reais:

```php
// Chaves de teste (para desenvolvimento)
define('STRIPE_PUBLISHABLE_KEY', 'pk_test_YOUR_PUBLISHABLE_KEY');
define('STRIPE_SECRET_KEY', 'sk_test_YOUR_SECRET_KEY');

// Chaves de produção (para ambiente live)
// define('STRIPE_PUBLISHABLE_KEY', 'pk_live_YOUR_PUBLISHABLE_KEY');
// define('STRIPE_SECRET_KEY', 'sk_live_YOUR_SECRET_KEY');
```

### 3. Atualizar o arquivo process_payment.php

Descomente e configure o código real do Stripe no arquivo `payment/process_payment.php`:

```php
// Remover a simulação e usar o código real:
require_once 'vendor/autoload.php';
\Stripe\Stripe::setApiKey(STRIPE_SECRET_KEY);

try {
    $charge = \Stripe\Charge::create([
        'amount' => $amount,
        'currency' => $currency,
        'description' => $description,
        'source' => $token,
        'metadata' => [
            'user_id' => $_SESSION['user_id'],
            'user_email' => $user['email']
        ]
    ]);
    
    $charge_id = $charge->id;
    $payment_success = true;
    
} catch(\Stripe\Exception\CardException $e) {
    echo json_encode(['success' => false, 'error' => $e->getError()->message]);
    exit;
} catch (\Stripe\Exception\RateLimitException $e) {
    echo json_encode(['success' => false, 'error' => 'Muitas requisições']);
    exit;
} catch (\Stripe\Exception\InvalidRequestException $e) {
    echo json_encode(['success' => false, 'error' => 'Parâmetros inválidos']);
    exit;
} catch (\Stripe\Exception\AuthenticationException $e) {
    echo json_encode(['success' => false, 'error' => 'Erro de autenticação']);
    exit;
} catch (\Stripe\Exception\ApiConnectionException $e) {
    echo json_encode(['success' => false, 'error' => 'Erro de conexão']);
    exit;
} catch (\Stripe\Exception\ApiErrorException $e) {
    echo json_encode(['success' => false, 'error' => 'Erro da API']);
    exit;
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'Erro desconhecido']);
    exit;
}
```

### 4. Atualizar a chave pública no JavaScript

No arquivo `payment/subscription.php`, substitua a chave pública:

```javascript
const stripe = Stripe('pk_test_YOUR_REAL_PUBLISHABLE_KEY');
```

## Funcionalidades Implementadas

### 1. Pagamento Único
- Cobrança de R$ 9,90 por mês
- Processamento via Stripe
- Validação de cartão em tempo real
- Feedback visual para o usuário

### 2. Registro de Pagamentos
- Tabela `payments` para histórico
- Status de assinatura do usuário
- Data de expiração da assinatura

### 3. Segurança
- Validação de sessão
- Sanitização de dados
- Tratamento de erros
- Logs de erro

## Estrutura do Banco de Dados

### Tabela payments
```sql
CREATE TABLE payments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    amount INT NOT NULL,
    currency VARCHAR(3) NOT NULL,
    stripe_charge_id VARCHAR(255),
    status VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### Campos adicionados na tabela users
```sql
ALTER TABLE users 
ADD COLUMN subscription_status VARCHAR(50) DEFAULT 'inactive',
ADD COLUMN subscription_expires_at TIMESTAMP NULL;
```

## Testes

### Cartões de Teste do Stripe

Para testar pagamentos, use estes números de cartão:

- **Sucesso**: 4242 4242 4242 4242
- **Cartão recusado**: 4000 0000 0000 0002
- **CVV inválido**: 4000 0000 0000 0127
- **Cartão expirado**: 4000 0000 0000 0069

Use qualquer data futura para expiração e qualquer CVV de 3 dígitos.

## Webhooks (Opcional)

Para receber notificações de eventos do Stripe, configure um webhook:

1. No dashboard do Stripe, vá em "Webhooks"
2. Adicione endpoint: `https://seudominio.com/webhook/stripe.php`
3. Selecione eventos: `charge.succeeded`, `charge.failed`

## Considerações de Produção

1. **SSL**: Sempre use HTTPS em produção
2. **Logs**: Implemente logs detalhados
3. **Backup**: Faça backup regular dos dados de pagamento
4. **Monitoramento**: Configure alertas para falhas de pagamento
5. **Compliance**: Certifique-se de estar em conformidade com PCI DSS

## Suporte

Para dúvidas sobre a integração Stripe:
- Documentação: https://stripe.com/docs
- Suporte: https://support.stripe.com

