from django.db import models
from django.contrib.auth.models import User
from controler.models import Mesa, RegistroDiario
from produtos.models import Produto
from decimal import Decimal


class Comanda(models.Model):
    STATUS_CHOICES = [
        ('aberta', 'Aberta'),
        ('fechada', 'Fechada'),
        ('cancelada', 'Cancelada'),
    ]

    mesa = models.ForeignKey(Mesa, on_delete=models.CASCADE, related_name='comandas')
    garcom = models.ForeignKey(User, on_delete=models.CASCADE, related_name='comandas')
    registro_diario = models.ForeignKey(RegistroDiario, on_delete=models.CASCADE, related_name='comandas')
    status = models.CharField(max_length=10, choices=STATUS_CHOICES, default='aberta')
    total = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    observacoes = models.TextField(blank=True, null=True)
    abertura = models.DateTimeField(auto_now_add=True)
    fechamento = models.DateTimeField(blank=True, null=True)
    arquivo_gerado = models.CharField(max_length=255, blank=True, null=True)

    class Meta:
        verbose_name = "Comanda"
        verbose_name_plural = "Comandas"
        ordering = ['-abertura']

    def __str__(self):
        return f"Comanda {self.id} - Mesa {self.mesa.numero} - {self.status}"

    def calcular_total(self):
        total = sum(item.subtotal for item in self.itens.all())
        self.total = total
        self.save()
        return total

    @property
    def nome_arquivo(self):
        if self.fechamento:
            return f"mesa_{self.mesa.numero}_garcom_{self.garcom.username}_grupo_{self.registro_diario.data.strftime('%Y%m%d')}_{self.fechamento.strftime('%H%M%S')}.txt"
        return None


class ComandaItem(models.Model):
    comanda = models.ForeignKey(Comanda, on_delete=models.CASCADE, related_name='itens')
    produto = models.ForeignKey(Produto, on_delete=models.CASCADE)
    quantidade = models.IntegerField(default=1)
    preco_unitario = models.DecimalField(max_digits=10, decimal_places=2)
    subtotal = models.DecimalField(max_digits=10, decimal_places=2)
    observacoes = models.TextField(blank=True, null=True)
    criado_em = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Item da Comanda"
        verbose_name_plural = "Itens da Comanda"

    def __str__(self):
        return f"{self.produto.nome} x{self.quantidade}"

    def save(self, *args, **kwargs):
        self.preco_unitario = self.produto.preco
        self.subtotal = self.preco_unitario * self.quantidade
        super().save(*args, **kwargs)
        # Recalcular total da comanda
        self.comanda.calcular_total()
