from django.db.models.signals import post_save, pre_save
from django.dispatch import receiver
from django.utils import timezone
from django.conf import settings
import os
from .models import Comanda, ComandaItem
from estoque.models import MovimentacaoEstoque


@receiver(post_save, sender=ComandaItem)
def baixar_estoque_item(sender, instance, created, **kwargs):
    """
    Signal para baixar estoque quando um item é adicionado à comanda
    """
    if created:
        produto = instance.produto
        quantidade_anterior = produto.estoque_atual
        
        # Reduzir estoque
        if produto.reduzir_estoque(instance.quantidade):
            # Registrar movimentação
            MovimentacaoEstoque.objects.create(
                produto=produto,
                tipo='saida',
                quantidade=instance.quantidade,
                quantidade_anterior=quantidade_anterior,
                quantidade_atual=produto.estoque_atual,
                motivo=f'Venda - Comanda {instance.comanda.id}',
                usuario=instance.comanda.garcom
            )


@receiver(pre_save, sender=Comanda)
def gerar_arquivo_comanda(sender, instance, **kwargs):
    """
    Signal para gerar arquivo TXT quando comanda é fechada
    """
    if instance.pk:  # Verifica se é uma atualização
        try:
            comanda_anterior = Comanda.objects.get(pk=instance.pk)
            # Se mudou de aberta para fechada
            if comanda_anterior.status == 'aberta' and instance.status == 'fechada':
                instance.fechamento = timezone.now()
                
                # Gerar arquivo TXT
                nome_arquivo = instance.nome_arquivo
                if nome_arquivo:
                    caminho_arquivo = os.path.join(settings.MEDIA_ROOT, 'comandas', nome_arquivo)
                    
                    # Criar diretório se não existir
                    os.makedirs(os.path.dirname(caminho_arquivo), exist_ok=True)
                    
                    # Gerar conteúdo do arquivo
                    conteudo = gerar_conteudo_arquivo(instance)
                    
                    # Salvar arquivo
                    with open(caminho_arquivo, 'w', encoding='utf-8') as arquivo:
                        arquivo.write(conteudo)
                    
                    instance.arquivo_gerado = nome_arquivo
                    
                    # Atualizar registro diário
                    registro = instance.registro_diario
                    registro.total_vendas += instance.total
                    registro.total_comandas += 1
                    registro.save()
                    
        except Comanda.DoesNotExist:
            pass


def gerar_conteudo_arquivo(comanda):
    """
    Gera o conteúdo do arquivo TXT da comanda
    """
    conteudo = []
    conteudo.append("=" * 50)
    conteudo.append("COMANDA - RESTAURANTE")
    conteudo.append("=" * 50)
    conteudo.append(f"Comanda: {comanda.id}")
    conteudo.append(f"Mesa: {comanda.mesa.numero}")
    conteudo.append(f"Garçom: {comanda.garcom.get_full_name() or comanda.garcom.username}")
    conteudo.append(f"Data: {comanda.registro_diario.data.strftime('%d/%m/%Y')}")
    conteudo.append(f"Abertura: {comanda.abertura.strftime('%H:%M:%S')}")
    conteudo.append(f"Fechamento: {comanda.fechamento.strftime('%H:%M:%S')}")
    conteudo.append("-" * 50)
    conteudo.append("ITENS:")
    conteudo.append("-" * 50)
    
    for item in comanda.itens.all():
        conteudo.append(f"{item.produto.nome}")
        conteudo.append(f"  Qtd: {item.quantidade} x R$ {item.preco_unitario:.2f} = R$ {item.subtotal:.2f}")
        if item.observacoes:
            conteudo.append(f"  Obs: {item.observacoes}")
        conteudo.append("")
    
    conteudo.append("-" * 50)
    conteudo.append(f"TOTAL: R$ {comanda.total:.2f}")
    conteudo.append("=" * 50)
    
    if comanda.observacoes:
        conteudo.append(f"Observações: {comanda.observacoes}")
        conteudo.append("=" * 50)
    
    return "\n".join(conteudo)
