from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib.auth.decorators import login_required
from django.urls import reverse_lazy
from django.http import HttpResponse, Http404
from django.conf import settings
from django.utils import timezone
from django.contrib import messages
import os
from .models import Comanda, ComandaItem
from controler.models import Mesa, RegistroDiario
from produtos.models import Produto


class ComandaListView(LoginRequiredMixin, ListView):
    model = Comanda
    template_name = 'comandas/list.html'
    context_object_name = 'comandas'
    paginate_by = 20

    def get_queryset(self):
        return Comanda.objects.select_related('mesa', 'garcom', 'registro_diario').order_by('-abertura')


class ComandaDetailView(LoginRequiredMixin, DetailView):
    model = Comanda
    template_name = 'comandas/detail.html'
    context_object_name = 'comanda'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['itens'] = self.object.itens.select_related('produto').all()
        return context


class ComandaCreateView(LoginRequiredMixin, CreateView):
    model = Comanda
    template_name = 'comandas/form.html'
    fields = ['mesa', 'observacoes']

    def form_valid(self, form):
        form.instance.garcom = self.request.user
        # Obter ou criar registro diário
        hoje = timezone.now().date()
        registro, created = RegistroDiario.objects.get_or_create(data=hoje)
        form.instance.registro_diario = registro
        return super().form_valid(form)

    def get_success_url(self):
        return reverse_lazy('comandas:detail', kwargs={'pk': self.object.pk})


class ComandaUpdateView(LoginRequiredMixin, UpdateView):
    model = Comanda
    template_name = 'comandas/form.html'
    fields = ['observacoes']

    def get_success_url(self):
        return reverse_lazy('comandas:detail', kwargs={'pk': self.object.pk})


class ComandaItemCreateView(LoginRequiredMixin, CreateView):
    model = ComandaItem
    template_name = 'comandas/item_form.html'
    fields = ['produto', 'quantidade', 'observacoes']

    def form_valid(self, form):
        comanda = get_object_or_404(Comanda, pk=self.kwargs['pk'])
        form.instance.comanda = comanda
        return super().form_valid(form)

    def get_success_url(self):
        return reverse_lazy('comandas:detail', kwargs={'pk': self.kwargs['pk']})


class ComandaItemDeleteView(LoginRequiredMixin, DeleteView):
    model = ComandaItem
    template_name = 'comandas/item_confirm_delete.html'

    def get_success_url(self):
        return reverse_lazy('comandas:detail', kwargs={'pk': self.object.comanda.pk})


@login_required
def fechar_comanda(request, pk):
    comanda = get_object_or_404(Comanda, pk=pk)
    
    if comanda.status == 'aberta':
        comanda.status = 'fechada'
        comanda.save()
        messages.success(request, 'Comanda fechada com sucesso!')
    else:
        messages.error(request, 'Esta comanda já foi fechada.')
    
    return redirect('comandas:detail', pk=pk)


class ArquivosComandaListView(LoginRequiredMixin, ListView):
    template_name = 'comandas/arquivos.html'
    context_object_name = 'arquivos'

    def get_queryset(self):
        comandas_path = os.path.join(settings.MEDIA_ROOT, 'comandas')
        arquivos = []
        
        if os.path.exists(comandas_path):
            for filename in os.listdir(comandas_path):
                if filename.endswith('.txt'):
                    filepath = os.path.join(comandas_path, filename)
                    stat = os.stat(filepath)
                    arquivos.append({
                        'nome': filename,
                        'tamanho': stat.st_size,
                        'data_criacao': timezone.datetime.fromtimestamp(stat.st_ctime)
                    })
        
        return sorted(arquivos, key=lambda x: x['data_criacao'], reverse=True)


@login_required
def download_arquivo(request, filename):
    filepath = os.path.join(settings.MEDIA_ROOT, 'comandas', filename)
    
    if not os.path.exists(filepath):
        raise Http404("Arquivo não encontrado")
    
    with open(filepath, 'rb') as f:
        response = HttpResponse(f.read(), content_type='text/plain')
        response['Content-Disposition'] = f'attachment; filename="{filename}"'
        return response
