from django.shortcuts import render, get_object_or_404, redirect
from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView, TemplateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib.auth.decorators import login_required
from django.urls import reverse_lazy
from django.utils import timezone
from django.contrib import messages
from .models import Mesa, RegistroDiario
from comandas.models import Comanda


class DashboardView(LoginRequiredMixin, TemplateView):
    template_name = 'controler/dashboard.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        hoje = timezone.now().date()
        
        # Obter ou criar registro diário
        registro, created = RegistroDiario.objects.get_or_create(data=hoje)
        
        context['registro_diario'] = registro
        context['comandas_abertas'] = Comanda.objects.filter(status='aberta').count()
        context['comandas_hoje'] = Comanda.objects.filter(registro_diario=registro).count()
        context['mesas_ocupadas'] = Mesa.objects.filter(comandas__status='aberta').distinct().count()
        
        return context


class MesaListView(LoginRequiredMixin, ListView):
    model = Mesa
    template_name = 'controler/mesas.html'
    context_object_name = 'mesas'

    def get_queryset(self):
        return Mesa.objects.filter(ativa=True).order_by('numero')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        # Adicionar informação de mesas ocupadas
        mesas_ocupadas = Mesa.objects.filter(comandas__status='aberta').values_list('id', flat=True)
        context['mesas_ocupadas'] = list(mesas_ocupadas)
        return context


class MesaCreateView(LoginRequiredMixin, CreateView):
    model = Mesa
    template_name = 'controler/mesa_form.html'
    fields = ['numero', 'descricao', 'capacidade']
    success_url = reverse_lazy('controler:mesas')


class MesaUpdateView(LoginRequiredMixin, UpdateView):
    model = Mesa
    template_name = 'controler/mesa_form.html'
    fields = ['numero', 'descricao', 'capacidade']
    success_url = reverse_lazy('controler:mesas')


class MesaDeleteView(LoginRequiredMixin, DeleteView):
    model = Mesa
    template_name = 'controler/mesa_confirm_delete.html'
    success_url = reverse_lazy('controler:mesas')


class RegistroDiarioView(LoginRequiredMixin, TemplateView):
    template_name = 'controler/registro_diario.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        hoje = timezone.now().date()
        
        # Obter ou criar registro diário
        registro, created = RegistroDiario.objects.get_or_create(data=hoje)
        
        context['registro'] = registro
        context['comandas'] = Comanda.objects.filter(registro_diario=registro).select_related('mesa', 'garcom')
        
        return context


@login_required
def arquivar_dia(request):
    hoje = timezone.now().date()
    
    try:
        registro = RegistroDiario.objects.get(data=hoje)
        
        # Verificar se há comandas abertas
        comandas_abertas = Comanda.objects.filter(registro_diario=registro, status='aberta').count()
        
        if comandas_abertas > 0:
            messages.error(request, f'Não é possível arquivar o dia. Existem {comandas_abertas} comandas ainda abertas.')
        else:
            registro.arquivado = True
            registro.save()
            messages.success(request, 'Dia arquivado com sucesso!')
            
    except RegistroDiario.DoesNotExist:
        messages.error(request, 'Registro diário não encontrado.')
    
    return redirect('controler:registro_diario')
