from django.shortcuts import render
from django.views.generic import ListView, CreateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from django.db import models
from produtos.models import Produto
from .models import MovimentacaoEstoque


class EstoqueListView(LoginRequiredMixin, ListView):
    model = Produto
    template_name = 'estoque/list.html'
    context_object_name = 'produtos'

    def get_queryset(self):
        return Produto.objects.filter(ativo=True).select_related('grupo').order_by('nome')


class EstoqueBaixoListView(LoginRequiredMixin, ListView):
    model = Produto
    template_name = 'estoque/baixo.html'
    context_object_name = 'produtos'

    def get_queryset(self):
        return Produto.objects.filter(
            ativo=True,
            estoque_atual__lte=models.F('estoque_minimo')
        ).select_related('grupo').order_by('estoque_atual')


class MovimentacaoEstoqueListView(LoginRequiredMixin, ListView):
    model = MovimentacaoEstoque
    template_name = 'estoque/movimentacoes.html'
    context_object_name = 'movimentacoes'
    paginate_by = 50

    def get_queryset(self):
        return MovimentacaoEstoque.objects.select_related('produto', 'usuario').order_by('-data_movimentacao')


class MovimentacaoEstoqueCreateView(LoginRequiredMixin, CreateView):
    model = MovimentacaoEstoque
    template_name = 'estoque/movimentacao_form.html'
    fields = ['produto', 'tipo', 'quantidade', 'motivo', 'observacoes']
    success_url = reverse_lazy('estoque:movimentacoes')

    def form_valid(self, form):
        form.instance.usuario = self.request.user
        form.instance.quantidade_anterior = form.instance.produto.estoque_atual
        
        if form.instance.tipo == 'entrada':
            form.instance.quantidade_atual = form.instance.quantidade_anterior + form.instance.quantidade
        elif form.instance.tipo == 'saida':
            form.instance.quantidade_atual = form.instance.quantidade_anterior - form.instance.quantidade
        else:  # ajuste
            form.instance.quantidade_atual = form.instance.quantidade
            
        return super().form_valid(form)
