<?php
/**
 * Dashboard Administrativo
 * Intimus RS - Visualização de Contatos
 */

require_once '../config.php';

// Verifica se o usuário está logado
if (!isset($_SESSION['admin_logado']) || $_SESSION['admin_logado'] !== true) {
    header('Location: login.php?erro=sessao');
    exit;
}

// Verifica timeout de sessão (30 minutos)
if (isset($_SESSION['admin_login_time']) && (time() - $_SESSION['admin_login_time'] > 1800)) {
    header('Location: logout.php');
    exit;
}

// Atualiza o tempo de atividade
$_SESSION['admin_login_time'] = time();

// Paginação
$limite = 20;
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina - 1) * $limite;

// Filtros de busca
$busca = isset($_GET['busca']) ? trim($_GET['busca']) : '';

try {
    $pdo = getDBConnection();
    
    // Query base
    $whereClause = '';
    $params = [];
    
    if (!empty($busca)) {
        $whereClause = " WHERE nome LIKE :busca OR telefone LIKE :busca OR endereco LIKE :busca";
        $params[':busca'] = '%' . $busca . '%';
    }
    
    // Conta total de registros
    $sqlCount = "SELECT COUNT(*) as total FROM contatos" . $whereClause;
    $stmtCount = $pdo->prepare($sqlCount);
    foreach ($params as $key => $value) {
        $stmtCount->bindValue($key, $value);
    }
    $stmtCount->execute();
    $totalRegistros = $stmtCount->fetch(PDO::FETCH_ASSOC)['total'];
    $totalPaginas = ceil($totalRegistros / $limite);
    
    // Busca os contatos
    $sql = "SELECT * FROM contatos" . $whereClause . " ORDER BY data_created DESC LIMIT :limite OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limite', $limite, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    
    $stmt->execute();
    $contatos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estatísticas
    $sqlStats = "SELECT 
                    COUNT(*) as total,
                    COUNT(CASE WHEN DATE(data_created) = CURDATE() THEN 1 END) as hoje,
                    COUNT(CASE WHEN WEEK(data_created) = WEEK(CURDATE()) THEN 1 END) as semana,
                    COUNT(CASE WHEN MONTH(data_created) = MONTH(CURDATE()) THEN 1 END) as mes
                 FROM contatos";
    $stmtStats = $pdo->query($sqlStats);
    $stats = $stmtStats->fetch(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Erro no dashboard: " . $e->getMessage());
    $contatos = [];
    $stats = ['total' => 0, 'hoje' => 0, 'semana' => 0, 'mes' => 0];
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Intimus RS</title>
    <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {
            --color-primary: #A91D5A;
            --color-secondary: #C72C57;
            --color-tertiary: #E6C3B6;
            --color-dark-bg: #464646;
            --color-light-text: #FFFFFF;
            --color-dark-text: #333333;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Montserrat', sans-serif;
            background-color: #f5f5f5;
            color: var(--color-dark-text);
        }

        .header {
            background-color: var(--color-primary);
            color: var(--color-light-text);
            padding: 20px 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .header-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header h1 {
            font-size: 1.8em;
            font-weight: 800;
        }

        .header-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .user-info {
            font-size: 0.95em;
        }

        .btn-logout {
            background-color: var(--color-light-text);
            color: var(--color-primary);
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-logout:hover {
            background-color: var(--color-tertiary);
            transform: translateY(-2px);
        }

        .container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 30px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background-color: var(--color-light-text);
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.08);
            text-align: center;
        }

        .stat-card h3 {
            color: #666;
            font-size: 0.9em;
            font-weight: 600;
            margin-bottom: 10px;
            text-transform: uppercase;
        }

        .stat-card .number {
            color: var(--color-primary);
            font-size: 2.5em;
            font-weight: 800;
        }

        .search-bar {
            background-color: var(--color-light-text);
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.08);
            margin-bottom: 20px;
        }

        .search-bar form {
            display: flex;
            gap: 15px;
        }

        .search-bar input {
            flex: 1;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 1em;
            font-family: 'Montserrat', sans-serif;
        }

        .search-bar button {
            padding: 12px 30px;
            background-color: var(--color-primary);
            color: var(--color-light-text);
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .search-bar button:hover {
            background-color: var(--color-secondary);
        }

        .table-container {
            background-color: var(--color-light-text);
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.08);
            overflow: hidden;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background-color: var(--color-primary);
            color: var(--color-light-text);
        }

        th {
            padding: 15px;
            text-align: left;
            font-weight: 700;
            font-size: 0.9em;
            text-transform: uppercase;
        }

        td {
            padding: 15px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 0.9em;
        }

        tbody tr:hover {
            background-color: #fafafa;
        }

        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 30px;
        }

        .pagination a,
        .pagination span {
            padding: 10px 15px;
            background-color: var(--color-light-text);
            border-radius: 8px;
            text-decoration: none;
            color: var(--color-dark-text);
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .pagination a:hover {
            background-color: var(--color-primary);
            color: var(--color-light-text);
        }

        .pagination .active {
            background-color: var(--color-primary);
            color: var(--color-light-text);
        }

        .no-data {
            text-align: center;
            padding: 40px;
            color: #999;
        }

        @media (max-width: 1200px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: 1fr;
            }

            .header-container {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }

            .search-bar form {
                flex-direction: column;
            }

            table {
                font-size: 0.85em;
            }

            th, td {
                padding: 10px;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-container">
            <h1>Dashboard Intimus RS</h1>
            <div class="header-info">
                <span class="user-info">Olá, <?php echo htmlspecialchars($_SESSION['admin_nome']); ?></span>
                <a href="logout.php" class="btn-logout">Sair</a>
            </div>
        </div>
    </header>

    <div class="container">
        <div class="stats-grid">
            <div class="stat-card">
                <h3>Total de Contatos</h3>
                <div class="number"><?php echo number_format($stats['total'], 0, ',', '.'); ?></div>
            </div>
            <div class="stat-card">
                <h3>Hoje</h3>
                <div class="number"><?php echo number_format($stats['hoje'], 0, ',', '.'); ?></div>
            </div>
            <div class="stat-card">
                <h3>Esta Semana</h3>
                <div class="number"><?php echo number_format($stats['semana'], 0, ',', '.'); ?></div>
            </div>
            <div class="stat-card">
                <h3>Este Mês</h3>
                <div class="number"><?php echo number_format($stats['mes'], 0, ',', '.'); ?></div>
            </div>
        </div>

        <div class="search-bar">
            <form method="get" action="">
                <input type="text" name="busca" placeholder="Buscar por nome, telefone ou endereço..." value="<?php echo htmlspecialchars($busca); ?>">
                <button type="submit">Buscar</button>
                <?php if (!empty($busca)): ?>
                    <a href="dashboard.php" style="padding: 12px 20px; background-color: #666; color: white; text-decoration: none; border-radius: 8px;">Limpar</a>
                <?php endif; ?>
            </form>
        </div>

        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nome</th>
                        <th>Telefone</th>
                        <th>Endereço</th>
                        <th>IP</th>
                        <th>Data</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($contatos) > 0): ?>
                        <?php foreach ($contatos as $contato): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($contato['id']); ?></td>
                                <td><?php echo htmlspecialchars($contato['nome']); ?></td>
                                <td><?php echo htmlspecialchars($contato['telefone']); ?></td>
                                <td><?php echo htmlspecialchars($contato['endereco']); ?></td>
                                <td><?php echo htmlspecialchars($contato['ip']); ?></td>
                                <td><?php echo date('d/m/Y H:i', strtotime($contato['data_created'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" class="no-data">Nenhum contato encontrado.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php if ($totalPaginas > 1): ?>
            <div class="pagination">
                <?php if ($pagina > 1): ?>
                    <a href="?pagina=<?php echo ($pagina - 1); ?><?php echo !empty($busca) ? '&busca=' . urlencode($busca) : ''; ?>">← Anterior</a>
                <?php endif; ?>

                <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
                    <?php if ($i == $pagina): ?>
                        <span class="active"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?pagina=<?php echo $i; ?><?php echo !empty($busca) ? '&busca=' . urlencode($busca) : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>

                <?php if ($pagina < $totalPaginas): ?>
                    <a href="?pagina=<?php echo ($pagina + 1); ?><?php echo !empty($busca) ? '&busca=' . urlencode($busca) : ''; ?>">Próxima →</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>

