# Projeto Django - E-commerce com Carrinho

## ✅ Status do Projeto

**Projeto verificado e pronto para produção**

Todos os componentes foram verificados:
- ✅ Models (Produto, Categoria, Pedido, ItemPedido)
- ✅ Forms (ProdutoForm, CategoriaForm)
- ✅ Views (Dashboard, Produtos, Carrinho)
- ✅ URLs (Todas as rotas configuradas)
- ✅ Carrinho de compras funcional
- ✅ Sistema de pedidos completo

---

## 📁 Estrutura do Projeto

```
projeto/
├── accounts/          # Autenticação e usuários
├── app/              # Configurações principais do Django
├── carrinho/         # Sistema de carrinho e pedidos
├── dashboard/        # Painel administrativo (CRUD de produtos)
├── loja/             # App de loja (se usado)
├── produtos/         # Models, Forms e Categorias
├── static/           # Arquivos estáticos
├── staticfiles/      # Arquivos estáticos coletados
├── media/            # Upload de imagens de produtos
├── manage.py         # Gerenciador Django
└── requirements.txt  # Dependências
```

---

## 🎯 Funcionalidades

### 1. **Gestão de Produtos** (Dashboard)
- ✅ Listar produtos com paginação e busca
- ✅ Criar produtos com todos os campos:
  - Nome, Descrição, Referência
  - Valor, Quantidade
  - **Unidade de Medida** (mg, g, kg, mm, cm, m, km, ml, l)
  - **Categoria** (relacionamento com tabela Categoria)
  - Imagem
  - Usuário cadastro (automático)
- ✅ Editar produtos
- ✅ Deletar produtos
- ✅ Controle de acesso (apenas staff)

### 2. **Gestão de Categorias**
- ✅ Listar categorias
- ✅ Criar categorias
- ✅ Editar categorias
- ✅ Deletar categorias
- ✅ Criação via AJAX (para uso em formulários)

### 3. **Carrinho de Compras**
- ✅ Adicionar produtos ao carrinho
- ✅ Remover produtos do carrinho
- ✅ Atualizar quantidades via AJAX
- ✅ Cálculo automático de totais
- ✅ Contador de itens no carrinho
- ✅ Suporte para usuários autenticados

### 4. **Sistema de Pedidos**
- ✅ Finalizar pedido
- ✅ Gerar transaction_id
- ✅ Status de pagamento
- ✅ Histórico de pedidos
- ✅ Visualização de detalhes do pedido
- ✅ Signal para gerar arquivo TXT do pedido

---

## 🔗 URLs Principais

### Carrinho e Loja
| URL | Função | Nome da Rota |
|-----|--------|--------------|
| `/` | Home (lista de produtos) | `home` |
| `/carrinho/` | Visualizar carrinho | `carrinho` |
| `/checkout/` | Finalizar compra | `checkout` |
| `/fechar_pedido/` | Processar pedido | `fechar_pedido` |
| `/atualizar_item/` | Adicionar/remover item (AJAX) | `atualizar_item` |

### Dashboard (Admin)
| URL | Função | Nome da Rota |
|-----|--------|--------------|
| `/dashboard/` | Home do dashboard | `dashboard:home` |
| `/dashboard/produtos/` | Listar produtos | `dashboard:produtos_list` |
| `/dashboard/produtos/criar/` | Criar produto | `dashboard:produto_create` |
| `/dashboard/produtos/<id>/editar/` | Editar produto | `dashboard:produto_update` |
| `/dashboard/produtos/<id>/deletar/` | Deletar produto | `dashboard:produto_delete` |
| `/dashboard/pedidos/` | Listar pedidos | `dashboard:pedidos_list` |
| `/dashboard/pedidos/<id>/` | Detalhes do pedido | `dashboard:pedido_detail` |

### Categorias
| URL | Função | Nome da Rota |
|-----|--------|--------------|
| `/produtos/categorias/` | Listar categorias | `lista_categorias` |
| `/produtos/categorias/criar/` | Criar categoria | `criar_categoria` |
| `/produtos/categorias/editar/<id>/` | Editar categoria | `editar_categoria` |
| `/produtos/categorias/deletar/<id>/` | Deletar categoria | `deletar_categoria` |

---

## 🚀 Instalação

### 1. Requisitos
- Python 3.8+
- Django 4.2+
- Pillow (para imagens)

### 2. Instalar Dependências
```bash
pip install -r requirements.txt
```

### 3. Configurar Banco de Dados
```bash
python manage.py makemigrations
python manage.py migrate
```

### 4. Criar Superusuário
```bash
python manage.py createsuperuser
```

### 5. Coletar Arquivos Estáticos
```bash
python manage.py collectstatic --noinput
```

### 6. Executar Servidor
```bash
python manage.py runserver
```

---

## 📊 Models Principais

### Produto
```python
- nome: CharField
- descricao: TextField
- quantidade: IntegerField
- valor: DecimalField
- imagem: ImageField
- ref: CharField (único)
- unidade_de_medida: CharField (choices)
- categoria: ForeignKey(Categoria)
- usuario_cadastro: ForeignKey(User)
- data_criacao: DateTimeField
- data_alteracao: DateTimeField
```

### Categoria
```python
- nome: CharField
- data_criacao: DateTimeField
```

### Pedido
```python
- usuario: ForeignKey(User)
- data_pedido: DateTimeField
- completo: BooleanField
- transaction_id: CharField
- status_pagamento: CharField
```

### ItemPedido
```python
- produto: ForeignKey(Produto)
- pedido: ForeignKey(Pedido)
- quantidade: IntegerField
- data_adicao: DateTimeField
```

---

## ⚙️ Configurações para Produção

### 1. **settings.py**
```python
DEBUG = False
ALLOWED_HOSTS = ['seu-dominio.com', 'www.seu-dominio.com']

# Configurar banco de dados de produção (PostgreSQL recomendado)
DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.postgresql',
        'NAME': 'seu_banco',
        'USER': 'seu_usuario',
        'PASSWORD': 'sua_senha',
        'HOST': 'localhost',
        'PORT': '5432',
    }
}

# Configurar arquivos estáticos
STATIC_ROOT = '/var/www/seu-projeto/static/'
MEDIA_ROOT = '/var/www/seu-projeto/media/'

# Segurança
SECRET_KEY = 'gerar-nova-chave-secreta'
SECURE_SSL_REDIRECT = True
SESSION_COOKIE_SECURE = True
CSRF_COOKIE_SECURE = True
```

### 2. **Servidor Web**
Recomendado: **Nginx + Gunicorn**

```bash
# Instalar Gunicorn
pip install gunicorn

# Executar com Gunicorn
gunicorn app.wsgi:application --bind 0.0.0.0:8000
```

### 3. **Variáveis de Ambiente**
Crie um arquivo `.env` para variáveis sensíveis:
```
SECRET_KEY=sua-chave-secreta
DEBUG=False
DATABASE_URL=postgresql://user:pass@localhost/dbname
```

---

## 🔒 Segurança

- ✅ Controle de acesso por usuário (login required)
- ✅ Controle de acesso para staff (dashboard)
- ✅ CSRF protection habilitado
- ✅ Validação de formulários
- ✅ Proteção contra SQL injection (ORM Django)
- ✅ Upload seguro de imagens

---

## 📝 Notas Importantes

### Carrinho
- O carrinho funciona apenas para usuários autenticados
- Itens são salvos no banco de dados
- Atualização em tempo real via AJAX
- Pedidos incompletos são mantidos até finalização

### Dashboard
- Acesso restrito a usuários staff (`is_staff=True`)
- Sistema de mensagens para feedback ao usuário
- Paginação automática (10 produtos por página)
- Busca por nome, descrição ou referência

### Produtos
- Imagens são salvas em `media/produtos_imagens/`
- Referência (ref) deve ser única
- Unidade de medida com 9 opções pré-definidas
- Categoria é opcional (pode ser null)

---

## 🐛 Troubleshooting

### Erro: "No such table"
```bash
python manage.py migrate
```

### Erro: "Static files not found"
```bash
python manage.py collectstatic
```

### Erro: "Permission denied" em media/
```bash
chmod -R 755 media/
```

### Carrinho não atualiza
- Verificar se usuário está autenticado
- Verificar console do navegador para erros JavaScript
- Verificar se URL `/atualizar_item/` está acessível

---

## 📞 Suporte

Para dúvidas ou problemas:
1. Verifique os logs do Django
2. Ative DEBUG=True temporariamente para ver erros detalhados
3. Verifique as migrations estão aplicadas
4. Confirme que todas as dependências estão instaladas

---

## ✅ Checklist de Deploy

- [ ] Atualizar SECRET_KEY
- [ ] Configurar DEBUG=False
- [ ] Configurar ALLOWED_HOSTS
- [ ] Configurar banco de dados de produção
- [ ] Executar migrations
- [ ] Coletar arquivos estáticos
- [ ] Configurar servidor web (Nginx/Apache)
- [ ] Configurar HTTPS/SSL
- [ ] Configurar backup automático do banco
- [ ] Testar todas as funcionalidades
- [ ] Criar usuário staff inicial

---

**Projeto verificado em**: 07/11/2025  
**Status**: ✅ Pronto para produção
