from django.db.models.signals import post_save
from django.dispatch import receiver
from produtos.models import Produto

@receiver(post_save, sender=Produto)
def gerar_ref_produto(sender, instance, created, **kwargs):
    """
    Gera automaticamente o campo ref do produto após salvar.
    Formato: codigo_barras(13) + descricao(160) + valor(10) + estoque(10) + unidade(4) + categoria(10)
    """
    # Evita loop infinito verificando se ref já foi gerado
    if instance.ref and not created:
        return
    
    # Código de barras: N 13 (Complementa zeros à esquerda)
    codigo_barras = str(instance.id).zfill(13)
    
    # Descrição do Produto: C 160 (Complementa Space à Direita)
    descricao = instance.nome[:160].ljust(160)
    
    # Valor do Produto: C 10 (Complementa zeros à esquerda) - em centavos
    valor = f"{int(instance.valor * 100):0>10}"
    
    # Estoque: C 10 (Complementa zeros à esquerda)
    estoque = str(instance.quantidade).zfill(10)
    
    # Unidade: C 4 (Complementa Space à Direita)
    unidade = instance.unidade_de_medida.upper().ljust(4)
    
    # Categoria: C 10 (Complementa zeros à esquerda)
    if instance.categoria:
        categoria = str(instance.categoria.id).zfill(10)
    else:
        categoria = "0000000000"
    
    # Gera o ref completo
    ref_gerado = f"{codigo_barras}{descricao}{valor}{estoque}{unidade}{categoria}"
    
    # Atualiza o campo ref sem disparar novo signal
    Produto.objects.filter(pk=instance.pk).update(ref=ref_gerado)
