import serial
import time
from django.shortcuts import render, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from .forms import ControleForm

# Configurações
# Windows: 'COM3', 'COM4', etc.
# Linux (Orange Pi/Raspberry): '/dev/ttyUSB0' ou '/dev/ttyACM0'
# Para listar portas no Linux: ls /dev/tty*
# Para dar permissão no Linux: sudo usermod -a -G dialout $USER (reinicie depois)
# PORTA_SERIAL = '/dev/ttyUSB0' # Descomente para Linux
PORTA_SERIAL = 'COM3'

BAUDRATE = 9600

# Variável global para manter a conexão aberta
_CONEXAO_SERIAL = None

def obter_conexao():
    """Mantém a porta serial aberta para evitar o delay de 2s e o reset do Arduino."""
    global _CONEXAO_SERIAL
    
    # Se já existir e estiver aberta, retorna ela
    if _CONEXAO_SERIAL and _CONEXAO_SERIAL.is_open:
        return _CONEXAO_SERIAL
        
    try:
        # Tenta abrir uma nova conexão
        _CONEXAO_SERIAL = serial.Serial(PORTA_SERIAL, BAUDRATE, timeout=1)
        # Delay de 2s necessário APENAS na primeira vez que abre (reset do bootloader)
        time.sleep(2)
        return _CONEXAO_SERIAL
    except Exception as e:
        print(f"Erro ao abrir serial: {e}")
        _CONEXAO_SERIAL = None
        return None

def enviar_comando(cmd):
    """Envia o comando usando a conexão persistente."""
    ser = obter_conexao()
    if ser:
        try:
            ser.write(cmd.encode())
            return True, None
        except Exception as e:
            # Se falhou, limpa a conexão para tentar reabrir na próxima
            global _CONEXAO_SERIAL
            _CONEXAO_SERIAL = None
            return False, f"Erro de envio: {e}"
    return False, "Porta serial não disponível. Verifique a conexão e se outro programa está usando a porta."

@login_required
def controle_view(request):
    if request.method == 'POST':
        form = ControleForm(request.POST)
        if form.is_valid():
            comando = form.cleaned_data['comando']
            sucesso, erro = enviar_comando(comando)
            
            if sucesso:
                # Opcional: Remova o message success se quiser que seja instantâneo sem feedback visual de texto
                # messages.success(request, f"Enviado: {comando}")
                pass
            else:
                messages.error(request, erro)
            
            return redirect('controle')
    else:
        form = ControleForm()

    return render(request, 'controle.html', {'form': form})
