import cv2
import threading

class VideoCamera(object):
    def __init__(self, index=0):
        # WINDOWS: cv2.CAP_DSHOW resolve problemas de 'can't grab frame'
        self.video = cv2.VideoCapture(index, cv2.CAP_DSHOW)
        
        # LINUX (Orange Pi/Raspberry):
        # O backend padrão (V4L2) geralmente funciona melhor.
        # Se der erro, tente forçar o V4L2:
        # self.video = cv2.VideoCapture(index, cv2.CAP_V4L2)
        # Ou simplesmente:
        # self.video = cv2.VideoCapture(index)

        self.index = index
        self.is_running = True
        self.lock = threading.Lock()
        
        # Tenta a primeira leitura para validar
        (self.grabbed, self.frame) = self.video.read()
        
        self.thread = threading.Thread(target=self.update, args=())
        self.thread.daemon = True
        self.thread.start()

    def stop(self):
        self.is_running = False
        if hasattr(self, 'video') and self.video.isOpened():
            self.video.release()

    def __del__(self):
        self.stop()

    def get_frame(self):
        with self.lock:
            if self.frame is None:
                return None
            ret, jpeg = cv2.imencode('.jpg', self.frame)
            return jpeg.tobytes()

    def update(self):
        error_count = 0
        while self.is_running:
            (grabbed, frame) = self.video.read()
            if not grabbed:
                error_count += 1
                if error_count > 10: # Se falhar muitas vezes, espera um pouco
                    threading.Event().wait(1.0)
                continue
            
            error_count = 0
            with self.lock:
                self.grabbed = grabbed
                self.frame = frame
